/** @file   menusettings.cpp
 * @brief   Implementation of MenuSettings - class.
 * @version $Revision: 1.7 $
 * @author  Tomi Lamminsaari
 */

#include "menusettings.h"
#include "GfxManager.h"
#include "soundsamples.h"
#include "settings.h"
#include "eng2d.h"
#include <string>
#include "helpnote.h"
#include "WWW_Defines.h"
using namespace eng2d;
using std::string;


//#define LEVEL_SELECTION 

namespace WeWantWar {


const int KItemIndexControls = 0;
const int KItemIndexMusic = 1;
#ifdef LEVEL_SELECTION
  const int KItemIndexStartFrom = 2;
  const int KItemIndexDifficulty = 3;
  const int KItemIndexExit = 4;
  const int MenuSettings::BACK_TO_MAINMENU = 4;
#else
  const int KItemIndexStartFrom = -1;
  const int KItemIndexDifficulty = 2;
  const int KItemIndexExit = 3;
  const int MenuSettings::BACK_TO_MAINMENU = 3;
#endif

#ifdef LEVEL_SELECTION
  const int KMenuItemCount = 5;
#else
  const int KMenuItemCount = 4;
#endif


/** Constructs new Settings-menu
 */
MenuSettings::MenuSettings() :
  MenuPage()
{
  // We create the hovering help items
  this->createHelpNotes();
  this->initGraphics();
}



/** Destructor
 * Param
 * Returns
 * Exception
 */
MenuSettings::~MenuSettings()
{
  this->cleanupGraphics();
}


/** Updates the Settings menu
 */
int MenuSettings::update()
{
  int ret = MenuPage::NO_PAGE_CHANGE;
  if ( this->getState() == MenuPage::STATE_FADEIN ) {
    ret = this->updateFadeInState();
    
  } else if ( this->getState() == MenuPage::STATE_VISIBLE ) {
    ret = this->updateVisibleState();
    if ( ret == MenuPage::PAGE_CHANGE ) {
      MenuPage::m_state = MenuPage::STATE_FADEOUT;
    }
    
  } else if ( this->getState() == MenuPage::STATE_FADEOUT ) {
    ret = this->updateFadeOutState();
    
  }
}


/** Redraws the settings menu
 */
void MenuSettings::redraw()
{
  this->updateSpriteBuffers();
  blit( GfxManager::titlePic, Display::buffer, 0,200, 0,200, 420,280 );
  blit( GfxManager::titlePic, Display::buffer, 400,390, 400,390, 230,80 );

  for ( int i=0; i < m_sprites.size(); i++ ) {
    Sprite* pS = m_sprites.at(i);
    pS->draw( Display::buffer );
  }
}


/** Creates the help notes
 */
void MenuSettings::createHelpNotes()
{
  std::vector<string>* pTT = new std::vector<string>();
  pTT->push_back( "Controlmethod you'd like to use." );
  pTT->push_back( "RELATIVE: Movement is relative to the" );
  pTT->push_back( "direction the man is facing" );
  pTT->push_back( "ABSOLUTE: Movement happens always either" );
  pTT->push_back( "vertically or horizontally or as combination" );
  pTT->push_back( "of them" );
  HelpNote* pItem = new HelpNote( 400, 390, 230, 80, *pTT );
  pItem->setOpacityRange( 0, 180 );
  this->addOverlayItem( pItem );
  delete pTT;
  
  pTT = new std::vector<string>();
  pTT->push_back( "Turns on or off the background musics" );
  pTT->push_back( "during the gameplay" );
  pItem = new HelpNote( 400, 390, 230, 80, *pTT );
  pItem->setOpacityRange( 0, 180 );
  this->addOverlayItem( pItem );
  delete pTT;
#ifdef LEVEL_SELECTION
  pTT = new std::vector<string>();
  pTT->push_back( "From which level you wish to begin" );
  pItem = new HelpNote( 400, 390, 230, 80, *pTT );
  pItem->setOpacityRange( 0, 180 );
  this->addOverlayItem( pItem );
  delete pTT;
#endif
  pTT = new std::vector<string>();
  pTT->push_back( "The difficulty level of the game" );
  pItem = new HelpNote( 400,390,230,80, *pTT );
  pItem->setOpacityRange( 0, 180 );
  this->addOverlayItem( pItem );
  delete pTT;
  
  pTT = new std::vector<string>();
  pTT->push_back( "Returns to mainmenu" );
  pItem = new HelpNote( 400, 390, 230, 80, *pTT );
  pItem->setOpacityRange( 0, 180 );
  this->addOverlayItem( pItem );
  delete pTT;
}


/** Checks the enter key
 */
int MenuSettings::checkEnterKey()
{
  if ( MenuPage::m_selection == BACK_TO_MAINMENU ) {
    return MenuPage::PAGE_CHANGE;
  }
  
  if ( MenuPage::m_selection == KItemIndexControls ) {
    // Toggle the controlmode
    if ( Settings::controlMode == Settings::CONTROLMODE1 ) {
      Settings::controlMode = Settings::CONTROLMODE2;
    } else {
      Settings::controlMode = Settings::CONTROLMODE1;
    }
    Sound::playSample(SMP_MENUSELECT, false);
    return MenuPage::NO_PAGE_CHANGE;
  }
  if ( MenuPage::m_selection == KItemIndexMusic ) {
    if ( Settings::musicOn ) {
      Settings::musicOn = false;
    } else {
      Settings::musicOn = true;
    }
    Sound::playSample(SMP_MENUSELECT, false);
    return MenuPage::NO_PAGE_CHANGE;
  }
#ifdef LEVEL_SELECTION
  if ( MenuPage::m_selection == KItemIndexStartFrom ) {
    Settings::startFromLevel += 1;
    if ( Settings::startFromLevel > 6 ) {
      Settings::startFromLevel = 0;
    }
    Sound::playSample( SMP_MENUSELECT, false );
    return MenuPage::NO_PAGE_CHANGE;
  }
#endif
  if ( MenuPage::m_selection == KItemIndexDifficulty ) {
    Settings::difficultyLevel += 1;
    if ( Settings::difficultyLevel > 2 ) {
      Settings::difficultyLevel = 1;
    }
    Sound::playSample( SMP_MENUSELECT, false );
    return MenuPage::NO_PAGE_CHANGE;
  }
  return MenuPage::NO_PAGE_CHANGE;
}



/** Inits the graphics
 */
int MenuSettings::initGraphics()
{
  // Controller settings
  float ycoord = 500;
  BITMAP* pTmp = create_bitmap( 240,24 );
  this->updateControllerBuffer( pTmp );
  Sprite* pS = new Sprite( pTmp );
  pS->position( Vec2D(150,ycoord) );
  m_sprites.push_back( pS );
  
  // Ingame music
  ycoord += 30;
  pTmp = create_bitmap( 240,24 );
  this->updateMusicsBuffer( pTmp );
  pS = new Sprite( pTmp );
  pS->position( Vec2D( 150,ycoord ) );
  m_sprites.push_back( pS );
  
  // start from level
#ifdef LEVEL_SELECTION
  ycoord += 30;
  pTmp = create_bitmap( 240,24 );
  this->updateStartFromBuffer( pTmp );
  pS = new Sprite( pTmp );
  pS->position( Vec2D( 150,ycoord) );
  m_sprites.push_back( pS );
#endif

  // Difficulty
  ycoord += 30;
  pTmp = create_bitmap( 240,24 );
  this->updateDifficultyBuffer( pTmp );
  pS = new Sprite( pTmp );
  pS->position( Vec2D( 150,ycoord ) );
  m_sprites.push_back( pS );
  
  // back to mainmenu
  ycoord += 30;
  pTmp = create_bitmap( 240,24 );
  this->updateExitBuffer( pTmp );
  pS = new Sprite( pTmp );
  pS->position( Vec2D( 150,ycoord) );
  m_sprites.push_back( pS );

  return 0;
}

/** Updates the graphics of Controller item.
 */
void MenuSettings::updateControllerBuffer( BITMAP* pB )
{
  clear_to_color( pB, makecol(255,0,255) );
  TPrinter out( pB );
  if ( MenuPage::m_selection == KItemIndexControls) {
    out.font( GfxManager::titleFont, "menuitems_sel" );
  } else {
    out.font( GfxManager::titleFont, "menuitems" );
  }
  string modetxt = "relative controls";
  if ( Settings::controlMode == Settings::CONTROLMODE2 ) {
    modetxt = "absolute controls";
  }
  out.print( modetxt );
}

/** Updates the ingame musics buffer
 */
void MenuSettings::updateMusicsBuffer( BITMAP* pB )
{
  clear_to_color( pB, makecol(255,0,255) );
  TPrinter out( pB );
  if ( MenuPage::m_selection == KItemIndexMusic ) {
    out.font( GfxManager::titleFont, "menuitems_sel" );
  } else {
    out.font( GfxManager::titleFont, "menuitems" );
  }
  string txt = "ingame music on";
  if ( Settings::musicOn == false ) {
    txt = "ingame music off";
  }
  out.print( txt );
}

/** Updates the start from level buffer
 */
void MenuSettings::updateStartFromBuffer( BITMAP* pB )
{
  clear_to_color( pB, makecol(255,0,255) );
  TPrinter out( pB );
  if ( MenuPage::m_selection == KItemIndexStartFrom ) {
    out.font( GfxManager::titleFont, "menuitems_sel" );
  } else {
    out.font( GfxManager::titleFont, "menuitems" );
  }
  out.print( "start from level " );
  out.print( Settings::startFromLevel );
}

/** Updates the difficulty buffer
 */
void MenuSettings::updateDifficultyBuffer( BITMAP* pB )
{
  clear_to_color( pB, makecol(255,0,255) );
  TPrinter out( pB );
  if ( MenuPage::m_selection == KItemIndexDifficulty ) {
    out.font( GfxManager::titleFont, "menuitems_sel" );
  } else {
    out.font( GfxManager::titleFont, "menuitems" );
  }
  string diffText = "difficulty easy";
  if ( Settings::difficultyLevel == Settings::KNormal ) {
    diffText = "difficulty normal";
  } else if ( Settings::difficultyLevel == Settings::KHard ) {
    diffText = "difficulty hard";
  }
  out.print( diffText );
}

/** Updates the exit item
 */
void MenuSettings::updateExitBuffer( BITMAP* pB )
{
  clear_to_color( pB, makecol(255,0,255) );
  TPrinter out( pB );
  if ( MenuPage::m_selection == KItemIndexExit ) {
    out.font( GfxManager::titleFont, "menuitems_sel" );
  } else {
    out.font( GfxManager::titleFont, "menuitems" );
  }
  out.print( "exit settings" );
}


/** Updates the sprite buffers
 */
void MenuSettings::updateSpriteBuffers()
{
  this->updateControllerBuffer( m_sprites.at(KItemIndexControls)->al_BITMAP() );
  this->updateMusicsBuffer( m_sprites.at(KItemIndexMusic)->al_BITMAP() );
#ifdef LEVEL_SELECTION
  this->updateStartFromBuffer( m_sprites.at(KItemIndexStartFrom)->al_BITMAP() );
#endif
  this->updateDifficultyBuffer( m_sprites.at(KItemIndexDifficulty)->al_BITMAP() );
  this->updateExitBuffer( m_sprites.at(KItemIndexExit)->al_BITMAP() );
}



/** Destroys the bitmaps
 */
void MenuSettings::cleanupGraphics()
{
  for ( int i=0; i < m_sprites.size(); i++ ) {
    if ( m_sprites.at(i) != 0 ) {
      if ( m_sprites.at(i)->al_BITMAP() != 0 ) {
        destroy_bitmap( m_sprites.at(i)->al_BITMAP() );
      }
      delete m_sprites.at(i);
    }
  }
  m_sprites.clear();
}



/** Updates the fadein state
 */
int MenuSettings::updateFadeInState()
{
  for ( int i=0; i < m_sprites.size(); i++ ) {
    Sprite* pS = m_sprites.at(i);
    Vec2D pos( pS->position() );
    float ny = pos.y() - 20;
    pos.y( ny );
    pS->position( pos );
  }
  Vec2D pos( m_sprites.at(0)->position() );
  if ( pos.y() < 240 ) {
    MenuPage::m_state = MenuPage::STATE_VISIBLE;
  }
  return MenuPage::NO_PAGE_CHANGE;
}



/** Updates the visible state
 */
int MenuSettings::updateVisibleState()
{
  // First we either fade in or fade out the help items
  int index = 0;
  while ( true ) {
    OverlayItem* pItem = this->getOverlayItem( index );
    if ( pItem == 0 ) {
      break;
    }
    if ( index == this->getSelection() ) {
      // This menuitem is selection. We fade it visible.
      pItem->fadeIn( 8 );

    } else {
      // This item is not selected. We fade it out
      pItem->fadeOut( 4 );
    }

    index += 1;
  }


  // Check the keyboard
  int ret = MenuPage::NO_PAGE_CHANGE;
  if ( key[MenuPage::s_lastKey] ) {
    return ret;
  }
  MenuPage::s_lastKey = KEY_O;

  if ( key[KEY_UP] ) {
    MenuPage::s_lastKey = KEY_UP;
    MenuPage::m_selection -= 1;
    if ( MenuPage::m_selection < 0 ) {
      MenuPage::m_selection = KMenuItemCount - 1;
    }
    Sound::playSample(SMP_MENUITEM, false);

  } else if ( key[KEY_DOWN] ) {
    MenuPage::s_lastKey = KEY_DOWN;
    MenuPage::m_selection += 1;
    if ( MenuPage::m_selection > (KMenuItemCount-1) ) {
      MenuPage::m_selection = 0;
    }
    Sound::playSample(SMP_MENUITEM, false);

  } else if ( key[KEY_SPACE] ) {
    MenuPage::s_lastKey = KEY_SPACE;
    ret = this->checkEnterKey();

  } else if ( key[KEY_ENTER] ) {
    MenuPage::s_lastKey = KEY_ENTER;
    ret = this->checkEnterKey();

  }
  if ( ret == MenuPage::PAGE_CHANGE ) {
    Sound::playSample(SMP_MENUSELECT, false);
  }

  if ( key[KEY_ESC] ) {
    MenuPage::s_lastKey = KEY_ESC;
    MenuPage::m_selection = BACK_TO_MAINMENU;
    ret = this->checkEnterKey();

  } else if ( key[KEY_BACKSPACE] ) {
    MenuPage::s_lastKey = KEY_BACKSPACE;
    MenuPage::m_selection = BACK_TO_MAINMENU;
    ret = this->checkEnterKey();
  }
  return ret;
}



/** Updates the fadeout state
 */
int MenuSettings::updateFadeOutState()
{
  for ( int i=0; i < m_sprites.size(); i++ ) {
    Sprite* pS = m_sprites.at(i);
    Vec2D pos( pS->position() );
    float ny = pos.y() + 20;
    pos.y( ny );
    pS->position( pos );
  }
  Vec2D pos( m_sprites.at(0)->position() );
  if ( pos.y() > 500 ) {
    return MenuPage::PAGE_CHANGE;
  }
  return MenuPage::NO_PAGE_CHANGE;
}


} // end of namespace
